# WordPress Headless CMS Setup Guide

This guide will help you set up a WordPress instance to work as a headless CMS for Paul Agonda's PR portfolio.

## Prerequisites

- WordPress 5.0+ (latest version recommended)
- REST API enabled (default in WordPress 4.7+)
- Advanced Custom Fields (ACF) Pro or Free plugin
- Custom Post Type UI plugin (or code-based custom post types)

## Step 1: WordPress Installation

### Option A: Managed Hosting (Recommended for beginners)
- **WordPress.com Business Plan** - Includes REST API access
- **Kinsta, WP Engine, or Bluehost** - All include REST API

### Option B: Self-Hosted
- Use a hosting provider with WordPress pre-installed
- Or manually install WordPress on your server
- Ensure you have SSH/cPanel access

## Step 2: Install Required Plugins

### 1. Advanced Custom Fields (ACF)
- Go to: WordPress Admin → Plugins → Add New
- Search for "Advanced Custom Fields"
- Install and activate the free version (or Pro for advanced features)

### 2. Custom Post Type UI
- Go to: WordPress Admin → Plugins → Add New
- Search for "Custom Post Type UI"
- Install and activate

### 3. REST API Enhancement (Optional but recommended)
- Search for "REST API" in plugins
- Consider "REST API Toolbox" for additional features

## Step 3: Create Custom Post Types

### Option A: Using Custom Post Type UI Plugin
1. Go to: WordPress Admin → CPT UI → Add New
2. Create two custom post types:

#### Post Type 1: Portfolio Item
- Post Type Name: `portfolio_item`
- Singular Name: `Portfolio Item`
- Plural Name: `Portfolio Items`
- Leave defaults for other options
- Check: "Show in REST"
- Check: "REST API Base: portfolio_item"

#### Post Type 2: Press Release
- Post Type Name: `press_release`
- Singular Name: `Press Release`
- Plural Name: `Press Releases`
- Leave defaults for other options
- Check: "Show in REST"
- Check: "REST API Base: press_release"

### Option B: Code-Based (Add to functions.php)

```php
// Add Portfolio Post Type
function register_portfolio_post_type() {
    register_post_type( 'portfolio_item', array(
        'labels' => array(
            'name' => 'Portfolio Items',
            'singular_name' => 'Portfolio Item',
        ),
        'public' => true,
        'show_in_rest' => true,
        'supports' => array( 'title', 'content', 'excerpt', 'thumbnail' ),
        'menu_position' => 5,
        'menu_icon' => 'dashicons-briefcase',
    ));
}
add_action( 'init', 'register_portfolio_post_type' );

// Add Press Release Post Type
function register_press_release_post_type() {
    register_post_type( 'press_release', array(
        'labels' => array(
            'name' => 'Press Releases',
            'singular_name' => 'Press Release',
        ),
        'public' => true,
        'show_in_rest' => true,
        'supports' => array( 'title', 'content', 'excerpt', 'thumbnail' ),
        'menu_position' => 6,
        'menu_icon' => 'dashicons-media-text',
    ));
}
add_action( 'init', 'register_press_release_post_type' );
```

## Step 4: Add Custom Fields with ACF

### Create Field Groups for Portfolio Items

1. Go to: WordPress Admin → ACF → Field Groups → Add New
2. Create a new field group called "Portfolio Details"
3. Add these fields:
   - `client_name` (Text)
   - `challenge` (Text Area)
   - `results` (Text Area)
   - `project_images` (Gallery or Repeater)
4. Set "Display this field group if" → Post Type is "portfolio_item"
5. Publish

### Create Field Groups for Press Releases

1. Go to: WordPress Admin → ACF → Field Groups → Add New
2. Create a new field group called "Press Release Details"
3. Add these fields:
   - `publication_name` (Text)
   - `publication_date` (Date Picker)
   - `pdf_link` (URL)
4. Set "Display this field group if" → Post Type is "press_release"
5. Publish

## Step 5: Configure REST API

### Enable REST API (Usually default)
1. Go to: WordPress Admin → Settings → Permalinks
2. Ensure a URL structure is selected (not "Plain")
3. Click "Save Changes"

### Test REST API Access
Visit these URLs in your browser:
- `https://your-wordpress-site.com/wp-json/wp/v2/posts`
- `https://your-wordpress-site.com/wp-json/wp/v2/portfolio_item`
- `https://your-wordpress-site.com/wp-json/wp/v2/press_release`

You should see JSON data returned.

### Enable CORS (if needed for local development)
If testing locally with a different domain, you may need CORS headers. Add to wp-config.php or use a CORS plugin.

## Step 6: Create Sample Content

### Create a Portfolio Item
1. Go to: WordPress Admin → Portfolio Items → Add New
2. Add title and content
3. Add featured image
4. Fill in custom fields (Client Name, Challenge, Results)
5. Publish

### Create a Press Release
1. Go to: WordPress Admin → Press Releases → Add New
2. Add title and content
3. Add featured image
4. Fill in custom fields (Publication Name, Publication Date, PDF Link)
5. Publish

### Create Blog Posts
1. Go to: WordPress Admin → Posts → Add New
2. Add title, content, and featured image
3. Publish

## Step 7: Configure Next.js Environment

Create a `.env.local` file in your Next.js project root:

```env
NEXT_PUBLIC_WORDPRESS_URL=https://your-wordpress-site.com
```

Replace `https://your-wordpress-site.com` with your actual WordPress URL.

## Step 8: Test the Integration

Run your Next.js development server:
```bash
npm run dev
```

Visit `http://localhost:3000` and check:
- Home page loads without errors
- Portfolio items appear on `/portfolio` page
- Blog posts appear on `/articles` page
- Press releases appear on `/press-releases` page

Check the browser console for any error messages.

## Troubleshooting

### No content appears
1. Check WordPress URL in `.env.local`
2. Verify REST API is accessible at `/wp-json/wp/v2/posts`
3. Check browser console for network errors
4. Ensure custom post types have `show_in_rest: true`

### CORS errors
1. Your WordPress and Next.js frontend are on different domains
2. Install "CORS" plugin on WordPress: https://wordpress.org/plugins/allow-json-cors/
3. Or add this to WordPress functions.php:
```php
add_filter( 'rest_allowed_cors_origins', function( $origins ) {
    $origins[] = 'http://localhost:3000';
    $origins[] = 'https://your-nextjs-domain.com';
    return $origins;
});
```

### Images not loading
1. Ensure featured images are uploaded and published
2. Check that image URLs are publicly accessible
3. Verify media is not private/protected

### Custom fields not appearing
1. Verify ACF field groups are assigned to correct post types
2. Ensure fields are published
3. Check ACF return format (should return object/array)

## API Endpoints Reference

```
GET  /wp-json/wp/v2/posts
GET  /wp-json/wp/v2/posts/{id}
GET  /wp-json/wp/v2/posts?slug={slug}

GET  /wp-json/wp/v2/portfolio_item
GET  /wp-json/wp/v2/portfolio_item/{id}
GET  /wp-json/wp/v2/portfolio_item?slug={slug}

GET  /wp-json/wp/v2/press_release
GET  /wp-json/wp/v2/press_release/{id}
GET  /wp-json/wp/v2/press_release?slug={slug}

GET  /wp-json/wp/v2/media
GET  /wp-json/wp/v2/categories
GET  /wp-json/wp/v2/tags
```

## Next Steps

1. Set up WordPress following this guide
2. Add your WordPress URL to `.env.local`
3. Create sample content in WordPress
4. Run `npm run dev` to test the integration
5. Customize content as needed
6. Deploy to Vercel

For questions or issues, refer to:
- [WordPress REST API Handbook](https://developer.wordpress.org/rest-api/)
- [ACF Documentation](https://www.advancedcustomfields.com/resources/)
